<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'customer') {
    header('Location: login.php');
    exit();
}

require_once 'admin/config/database.php';

$database = new Database();
$conn = $database->getConnection();

// Get user data
$user_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

if (!$user) {
    header('Location: login.php');
    exit();
}

// Get account statistics
$stmt = $conn->prepare("SELECT COUNT(*) as total_transactions FROM transactions WHERE user_id = ?");
$stmt->execute([$user_id]);
$transaction_count = $stmt->fetch()['total_transactions'];

$stmt = $conn->prepare("SELECT SUM(amount) as total_deposits FROM deposits WHERE user_id = ? AND status = 'Approved'");
$stmt->execute([$user_id]);
$total_deposits = $stmt->fetch()['total_deposits'] ?? 0;

$stmt = $conn->prepare("SELECT SUM(amount) as total_withdrawals FROM transactions WHERE user_id = ? AND transaction_type = 'Withdrawal' AND status = 'Completed'");
$stmt->execute([$user_id]);
$total_withdrawals = $stmt->fetch()['total_withdrawals'] ?? 0;

$database->closeConnection();

// Calculate age from date of birth
$age = null;
if ($user['date_of_birth']) {
    $birth_date = new DateTime($user['date_of_birth']);
    $today = new DateTime();
    $age = $today->diff($birth_date)->y;
}

// Format account creation date
$account_created = new DateTime($user['created_at']);
$days_since_creation = $today->diff($account_created)->days;

// Get current date and time
$current_date = date('l, F j, Y');
$current_time = date('H:i:s');
$greeting = 'Good Morning';
$hour = (int)date('H');
if ($hour >= 12 && $hour < 17) {
    $greeting = 'Good Afternoon';
} elseif ($hour >= 17) {
    $greeting = 'Good Evening';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Account Information - TrustBeam Credit</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .account-info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .info-card {
            background: white;
            border-radius: 15px;
            padding: 0;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border: 1px solid #e0e6ed;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            overflow: hidden;
        }

        .info-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }

        .info-card .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 25px;
            margin: 0;
        }

        .info-card .card-header h3 {
            margin: 0;
            font-size: 1.2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .info-card .card-header h3 i {
            font-size: 1.1rem;
        }

        .info-card .card-content {
            padding: 25px;
        }

        .info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 0;
            border-bottom: 1px solid #f1f3f4;
        }

        .info-row:last-child {
            border-bottom: none;
        }

        .info-label {
            font-weight: 600;
            color: #5a6c7d;
            font-size: 0.95rem;
        }

        .info-value {
            color: #2c3e50;
            font-weight: 500;
            text-align: right;
            max-width: 60%;
            word-break: break-word;
        }

        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            background: #e3f2fd;
            color: #1565c0;
        }

        .status-badge.active {
            background: #d4edda;
            color: #155724;
        }

        .profile-photo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid #e0e6ed;
        }

        .no-photo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #f8f9fa;
            display: flex;
            align-items: center;
            justify-content: center;
            border: 3px solid #e0e6ed;
            color: #6c757d;
            font-size: 1.5rem;
        }

        .page-header {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            border: 1px solid #e0e6ed;
        }

        .page-header .header-content h1 {
            color: #2c3e50;
            margin: 0 0 10px 0;
            font-size: 2rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .page-header .header-content h1 i {
            color: #667eea;
        }

        .page-header .header-content p {
            color: #5a6c7d;
            margin: 0;
            font-size: 1.1rem;
        }

        @media (max-width: 768px) {
            .account-info-grid {
                grid-template-columns: 1fr;
            }
            
            .info-row {
                flex-direction: column;
                align-items: flex-start;
                gap: 5px;
            }
            
            .info-value {
                text-align: left;
                max-width: 100%;
            }
        }
    </style>
</head>
<body class="dashboard-page">
    <!-- Top Header Bar -->
    <div class="dashboard-header">
        <div class="header-left">
            <div class="current-balance">
                <span class="balance-label">Current Balance</span>
                <span class="balance-amount">$<?php echo number_format($user['balance'], 0); ?></span>
            </div>
        </div>
        <div class="header-center">
            <div class="current-date"><?php echo $current_date; ?></div>
        </div>
        <div class="header-right">
            <div class="header-actions">
                <button class="notification-btn">
                    <i class="fas fa-bell"></i>
                    <span class="notification-badge">3</span>
                </button>
                <div class="user-profile">
                    <div class="user-avatar">
                        <?php if ($user['profile_photo']): ?>
                            <img src="admin/<?php echo htmlspecialchars($user['profile_photo']); ?>" alt="Profile Photo">
                        <?php else: ?>
                            <span><?php echo strtoupper(substr($user['firstname'], 0, 1) . substr($user['lastname'], 0, 1)); ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="user-info">
                        <span class="user-name"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></span>
                        <span class="user-id"><?php echo htmlspecialchars($user['account_number']); ?></span>
                        <span class="kyc-badge">KYC Verified</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="dashboard-container">
        <!-- Mobile Menu Toggle -->
        <button class="mobile-menu-toggle" id="mobileMenuToggle">
            <i class="fas fa-bars"></i>
        </button>
        
        <!-- Sidebar Overlay -->
        <div class="sidebar-overlay" id="sidebarOverlay"></div>
        
        <!-- Left Sidebar -->
        <aside class="dashboard-sidebar">
            <div class="sidebar-header">
                <div class="logo">
                    <div class="logo-emblem">
                        <span class="logo-t">T</span>
                        <span class="logo-b">B</span>
                    </div>
                    <div class="logo-text">
                        <div class="logo-main">TrustBeam</div>
                        <div class="logo-sub">Credit</div>
                    </div>
                </div>
            </div>
            
            <div class="user-profile-sidebar">
                <div class="user-avatar-sidebar">
                    <?php if ($user['profile_photo']): ?>
                        <img src="admin/<?php echo htmlspecialchars($user['profile_photo']); ?>" alt="Profile Photo">
                    <?php else: ?>
                        <span><?php echo strtoupper(substr($user['firstname'], 0, 1) . substr($user['lastname'], 0, 1)); ?></span>
                    <?php endif; ?>
                </div>
                <div class="user-details">
                    <div class="user-name"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></div>
                    <div class="user-id"><?php echo htmlspecialchars($user['account_number']); ?></div>
                    <span class="kyc-badge">KYC Verified</span>
                </div>
                <div class="profile-actions">
                    <a href="profile.php" class="btn btn-profile">Profile</a>
                    <a href="logout.php" class="btn btn-logout">Logout</a>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-section">
                    <h3>MAIN MENU</h3>
                    <ul>
                        <li>
                            <a href="dashboard.php">
                                <i class="fas fa-home"></i>
                                <span>Dashboard</span>
                            </a>
                        </li>
                        <li>
                            <a href="transactions.php">
                                <i class="fas fa-exchange-alt"></i>
                                <span>Transactions</span>
                            </a>
                        </li>
                        <li>
                            <a href="cards.php">
                                <i class="fas fa-credit-card"></i>
                                <span>Cards</span>
                            </a>
                        </li>
                    </ul>
                </div>
                
                <div class="nav-section">
                    <h3>TRANSFERS</h3>
                    <ul>
                        <li>
                            <a href="transfer.php">
                                <i class="fas fa-paper-plane"></i>
                                <span>Transfer Funds</span>
                            </a>
                        </li>
                        <li>
                            <a href="deposit.php">
                                <i class="fas fa-arrow-down"></i>
                                <span>Deposit</span>
                            </a>
                        </li>
                    </ul>
                </div>
                
                <div class="nav-section">
                    <h3>SERVICES</h3>
                    <ul>
                        <li>
                            <a href="loan-request.php">
                                <i class="fas fa-hand-holding-usd"></i>
                                <span>Loan Request</span>
                            </a>
                        </li>
                        <li>
                            <a href="irs-refund.php">
                                <i class="fas fa-file-invoice-dollar"></i>
                                <span>IRS Tax Refund</span>
                            </a>
                        </li>
                    </ul>
                </div>
                
                <div class="nav-section">
                    <h3>ACCOUNT</h3>
                    <ul>
                        <li class="active">
                            <a href="account-info.php">
                                <i class="fas fa-user-circle"></i>
                                <span>Account Info</span>
                            </a>
                        </li>
                        <li>
                            <a href="loan-history.php">
                                <i class="fas fa-history"></i>
                                <span>Loan History</span>
                            </a>
                        </li>
                        <li>
                            <a href="settings.php">
                                <i class="fas fa-cog"></i>
                                <span>Settings</span>
                            </a>
                        </li>
                        <li>
                            <a href="support.php">
                                <i class="fas fa-headset"></i>
                                <span>Support Ticket</span>
                            </a>
                        </li>
                    </ul>
                </div>
            </nav>
            
            <div class="sidebar-footer">
                <div class="secure-banking">
                    <i class="fas fa-shield-alt"></i>
                    <span>Secure Banking</span>
                </div>
                <div class="version">v1.2.0</div>
            </div>
        </aside>

        <!-- Main Content Area -->
        <main class="dashboard-main">
            <!-- Page Header -->
            <div class="page-header">
                <div class="header-content">
                    <h1><i class="fas fa-user-circle"></i> Account Information</h1>
                    <p>Complete overview of your account details and activity</p>
                </div>
            </div>

            <!-- Account Statistics -->
            <div class="overview-cards">
                <div class="overview-card">
                    <div class="card-icon">
                        <i class="fas fa-exchange-alt"></i>
                    </div>
                    <div class="card-content">
                        <h3>Total Transactions</h3>
                        <div class="card-amount"><?php echo number_format($transaction_count); ?></div>
                    </div>
                </div>
                
                <div class="overview-card">
                    <div class="card-icon income">
                        <i class="fas fa-arrow-up"></i>
                    </div>
                    <div class="card-content">
                        <h3>Total Deposits</h3>
                        <div class="card-amount">$<?php echo number_format($total_deposits, 2); ?></div>
                    </div>
                </div>
                
                <div class="overview-card">
                    <div class="card-icon outgoing">
                        <i class="fas fa-arrow-down"></i>
                    </div>
                    <div class="card-content">
                        <h3>Total Withdrawals</h3>
                        <div class="card-amount">$<?php echo number_format($total_withdrawals, 2); ?></div>
                    </div>
                </div>
                
                <div class="overview-card">
                    <div class="card-icon limit">
                        <i class="fas fa-calendar"></i>
                    </div>
                    <div class="card-content">
                        <h3>Days as Member</h3>
                        <div class="card-amount"><?php echo $days_since_creation; ?></div>
                    </div>
                </div>
            </div>

            <!-- Account Information Cards -->
            <div class="account-info-grid">
                <!-- Personal Information -->
                <div class="info-card">
                    <div class="card-header">
                        <h3><i class="fas fa-user"></i> Personal Information</h3>
                    </div>
                    <div class="card-content">
                        <div class="info-row">
                            <span class="info-label">Full Name:</span>
                            <span class="info-value">
                                <?php echo htmlspecialchars($user['firstname'] . ' ' . ($user['middlename'] ? $user['middlename'] . ' ' : '') . $user['lastname']); ?>
                            </span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Username:</span>
                            <span class="info-value"><?php echo htmlspecialchars($user['username']); ?></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Email:</span>
                            <span class="info-value"><?php echo htmlspecialchars($user['email']); ?></span>
                        </div>
                        
                        <?php if ($user['phone']): ?>
                        <div class="info-row">
                            <span class="info-label">Phone:</span>
                            <span class="info-value"><?php echo htmlspecialchars($user['phone']); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($user['date_of_birth']): ?>
                        <div class="info-row">
                            <span class="info-label">Date of Birth:</span>
                            <span class="info-value">
                                <?php echo date('M j, Y', strtotime($user['date_of_birth'])); ?>
                                <?php if ($age): ?>
                                    (<?php echo $age; ?> years old)
                                <?php endif; ?>
                            </span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($user['nationality']): ?>
                        <div class="info-row">
                            <span class="info-label">Nationality:</span>
                            <span class="info-value"><?php echo htmlspecialchars($user['nationality']); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($user['address']): ?>
                        <div class="info-row">
                            <span class="info-label">Address:</span>
                            <span class="info-value"><?php echo htmlspecialchars($user['address']); ?></span>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Account Details -->
                <div class="info-card">
                    <div class="card-header">
                        <h3><i class="fas fa-university"></i> Account Details</h3>
                    </div>
                    <div class="card-content">
                        <div class="info-row">
                            <span class="info-label">Account Number:</span>
                            <span class="info-value"><?php echo htmlspecialchars($user['account_number']); ?></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Account Type:</span>
                            <span class="info-value">
                                <span class="status-badge"><?php echo htmlspecialchars($user['account_type']); ?></span>
                            </span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Account Status:</span>
                            <span class="info-value">
                                <span class="status-badge active"><?php echo htmlspecialchars($user['status']); ?></span>
                            </span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Member Since:</span>
                            <span class="info-value"><?php echo date('M j, Y', strtotime($user['created_at'])); ?></span>
                        </div>
                        
                        <?php if ($user['last_login']): ?>
                        <div class="info-row">
                            <span class="info-label">Last Login:</span>
                            <span class="info-value"><?php echo date('M j, Y g:i A', strtotime($user['last_login'])); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="info-row">
                            <span class="info-label">Profile Photo:</span>
                            <span class="info-value">
                                <?php if ($user['profile_photo']): ?>
                                    <img src="admin/uploads/profiles/<?php echo htmlspecialchars($user['profile_photo']); ?>" 
                                         alt="Profile Photo" class="profile-photo">
                                <?php else: ?>
                                    <div class="no-photo">
                                        <i class="fas fa-user"></i>
                                    </div>
                                <?php endif; ?>
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Account Activity -->
                <div class="info-card">
                    <div class="card-header">
                        <h3><i class="fas fa-chart-line"></i> Account Activity</h3>
                    </div>
                    <div class="card-content">
                        <div class="info-row">
                            <span class="info-label">Account Created:</span>
                            <span class="info-value"><?php echo date('M j, Y g:i A', strtotime($user['created_at'])); ?></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Last Updated:</span>
                            <span class="info-value"><?php echo date('M j, Y g:i A', strtotime($user['updated_at'])); ?></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Days Active:</span>
                            <span class="info-value"><?php echo $days_since_creation; ?> days</span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Total Transactions:</span>
                            <span class="info-value"><?php echo number_format($transaction_count); ?></span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">Account Value:</span>
                            <span class="info-value">$<?php echo number_format($user['balance'], 2); ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script src="assets/js/main.js"></script>
</body>
</html>
