<?php
session_start();
require_once 'config/database.php';
require_once __DIR__ . '/lib/EmailService.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

$database = new Database();
$conn = $database->getConnection();

// Get admin info
$admin_id = $_SESSION['admin_id'];
$stmt = $conn->prepare("SELECT * FROM admins WHERE id = ?");
$stmt->execute([$admin_id]);
$admin = $stmt->fetch();

// Get dashboard statistics
$stmt = $conn->query("SELECT COUNT(*) as total_users FROM users");
$total_users = $stmt->fetch()['total_users'];

$stmt = $conn->query("SELECT COUNT(*) as pending_users FROM users WHERE status = 'Pending'");
$pending_users = $stmt->fetch()['pending_users'];

$stmt = $conn->query("SELECT COUNT(*) as total_transactions FROM transactions");
$total_transactions = $stmt->fetch()['total_transactions'];

$stmt = $conn->query("SELECT COUNT(*) as pending_loans FROM loan_applications WHERE status = 'Pending'");
$pending_loans = $stmt->fetch()['pending_loans'];

$smtp_status = 'Unknown';
$site_name = 'TrustBeam Credit';
try {
    $sys = $conn->query("SELECT * FROM system_settings WHERE id = 1")->fetch();
    $gen = $conn->query("SELECT * FROM general_settings WHERE id = 1")->fetch();
    if ($gen && !empty($gen['site_name'])) { $site_name = $gen['site_name']; }
    if ($sys) {
        $smtp_status = ((int)$sys['smtp_enabled'] === 1 && !empty($sys['smtp_host'])) ? 'Configured' : 'Not configured';
    } else {
        $smtp_status = 'Not configured';
    }
} catch (\Throwable $t) {
    $smtp_status = 'Not configured';
}

$database->closeConnection();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TrustBeam Credit - Admin Dashboard</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="admin-container">
        <!-- Header -->
        <header class="admin-header">
            <div class="header-left">
                <h1 class="logo">TrustBeam Credit</h1>
                <button class="menu-toggle" id="menuToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <div class="search-bar">
                    <i class="fas fa-search"></i>
                    <input type="text" placeholder="Search...">
                </div>
            </div>
            <div class="header-right">
                <div class="admin-profile">
                    <span>Welcome, <?php echo htmlspecialchars($admin['username']); ?></span>
                    <div class="profile-dropdown">
                        <i class="fas fa-user"></i>
                        <div class="dropdown-content">
                            <a href="profile.php">Profile</a>
                            <a href="logout.php">Logout</a>
                        </div>
                    </div>
                </div>
            </div>
        </header>

        <!-- Sidebar -->
        <aside class="admin-sidebar" id="sidebar">
            <div class="sidebar-header">
                <h3>Admin Manager</h3>
                <p>Admin</p>
            </div>
            <nav class="sidebar-nav">
                <ul>
                    <li class="active">
                        <a href="index.php">
                            <i class="fas fa-home"></i>
                            <span>Dashboard</span>
                        </a>
                    </li>
                    <li>
                        <a href="users.php">
                            <i class="fas fa-users"></i>
                            <span>Manage Users</span>
                        </a>
                    </li>
                    <li>
                        <a href="create-user.php">
                            <i class="fas fa-user-plus"></i>
                            <span>Create New User</span>
                        </a>
                    </li>
                    <li>
                        <a href="applications.php">
                            <i class="fas fa-user-friends"></i>
                            <span>New User Application(s)</span>
                        </a>
                    </li>
                    <li>
                        <a href="transactions.php">
                            <i class="fas fa-exchange-alt"></i>
                            <span>Transfer Transactions</span>
                        </a>
                    </li>
                    <li>
                        <a href="deposits.php">
                            <i class="fas fa-arrow-down"></i>
                            <span>Users Deposits</span>
                        </a>
                    </li>
                    <li class="has-submenu">
                        <a href="#" class="submenu-toggle">
                            <i class="fas fa-cog"></i>
                            <span>Loan Applications</span>
                            <i class="fas fa-chevron-down"></i>
                        </a>
                        <ul class="submenu">
                            <li><a href="loans.php">View All</a></li>
                            <li><a href="loans.php?status=pending">Pending</a></li>
                            <li><a href="loans.php?status=approved">Approved</a></li>
                        </ul>
                    </li>
                    <li class="has-submenu">
                        <a href="#" class="submenu-toggle">
                            <i class="fas fa-credit-card"></i>
                            <span>Virtual Cards</span>
                            <i class="fas fa-chevron-down"></i>
                        </a>
                        <ul class="submenu">
                            <li><a href="cards.php">Manage Cards</a></li>
                            <li><a href="cards.php?status=active">Active</a></li>
                        </ul>
                    </li>
                    <li class="has-submenu">
                        <a href="#" class="submenu-toggle">
                            <i class="fas fa-file-invoice-dollar"></i>
                            <span>IRS Refunds</span>
                            <i class="fas fa-chevron-down"></i>
                        </a>
                        <ul class="submenu">
                            <li><a href="refunds.php">View All</a></li>
                            <li><a href="refunds.php?status=pending">Pending</a></li>
                        </ul>
                    </li>
                    <li>
                        <a href="email.php">
                            <i class="fas fa-envelope"></i>
                            <span>Email Services</span>
                        </a>
                    </li>
                    <li class="has-submenu">
                        <a href="#" class="submenu-toggle">
                            <i class="fas fa-user-shield"></i>
                            <span>Administrator(s)</span>
                            <i class="fas fa-chevron-down"></i>
                        </a>
                        <ul class="submenu">
                            <li><a href="admins.php">View All</a></li>
                            <li><a href="create-admin.php">Add New</a></li>
                        </ul>
                    </li>
                    <li class="has-submenu">
                        <a href="#" class="submenu-toggle">
                            <i class="fas fa-cog"></i>
                            <span>Settings</span>
                            <i class="fas fa-chevron-down"></i>
                        </a>
                        <ul class="submenu">
                            <li><a href="settings.php">General</a></li>
                            <li><a href="security.php">Security</a></li>
                        </ul>
                    </li>
                </ul>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="admin-main">
            <div class="dashboard-header">
                <h2>Dashboard</h2>
                <p>Welcome back! Here's what's happening with your bank today.</p>
            </div>

            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($total_users); ?></h3>
                        <p>Total Users</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-user-clock"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($pending_users); ?></h3>
                        <p>Pending Users</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-exchange-alt"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($total_transactions); ?></h3>
                        <p>Total Transactions</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-hand-holding-usd"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($pending_loans); ?></h3>
                        <p>Pending Loans</p>
                    </div>
                </div>
            </div>

            <!-- Recent Activity -->
            <div class="dashboard-sections">
                <div class="section-card">
                    <div class="section-header">
                        <h3>Recent Users</h3>
                        <a href="users.php" class="view-all">View All</a>
                    </div>
                    <div class="recent-users">
                        <?php
                        $database = new Database();
                        $conn = $database->getConnection();
                        $stmt = $conn->query("SELECT * FROM users ORDER BY created_at DESC LIMIT 5");
                        $recent_users = $stmt->fetchAll();
                        $database->closeConnection();
                        
                        if (empty($recent_users)): ?>
                            <p class="no-data">No recent users found.</p>
                        <?php else: ?>
                            <?php foreach ($recent_users as $user): ?>
                                <div class="user-item">
                                    <div class="user-avatar">
                                        <i class="fas fa-user"></i>
                                    </div>
                                    <div class="user-info">
                                        <h4><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></h4>
                                        <p><?php echo htmlspecialchars($user['email']); ?></p>
                                        <span class="user-status <?php echo strtolower($user['status']); ?>">
                                            <?php echo htmlspecialchars($user['status']); ?>
                                        </span>
                                    </div>
                                    <div class="user-actions">
                                        <a href="users.php?id=<?php echo $user['id']; ?>" class="btn-view">View</a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="section-card">
                    <div class="section-header">
                        <h3>Recent Transactions</h3>
                        <a href="transactions.php" class="view-all">View All</a>
                    </div>
                    <div class="recent-transactions">
                        <?php
                        $database = new Database();
                        $conn = $database->getConnection();
                        $stmt = $conn->query("SELECT t.*, u.firstname, u.lastname FROM transactions t 
                                           LEFT JOIN users u ON t.user_id = u.id 
                                           ORDER BY t.created_at DESC LIMIT 5");
                        $recent_transactions = $stmt->fetchAll();
                        $database->closeConnection();
                        
                        if (empty($recent_transactions)): ?>
                            <p class="no-data">No recent transactions found.</p>
                        <?php else: ?>
                            <?php foreach ($recent_transactions as $transaction): ?>
                                <div class="transaction-item">
                                    <div class="transaction-icon">
                                        <i class="fas fa-<?php echo $transaction['transaction_type'] === 'Deposit' ? 'arrow-down' : 'arrow-up'; ?>"></i>
                                    </div>
                                    <div class="transaction-info">
                                        <h4><?php echo htmlspecialchars($transaction['transaction_type']); ?></h4>
                                        <p><?php echo htmlspecialchars($transaction['firstname'] . ' ' . $transaction['lastname']); ?></p>
                                        <span class="transaction-amount">
                                            $<?php echo number_format($transaction['amount'], 2); ?>
                                        </span>
                                    </div>
                                    <div class="transaction-status <?php echo strtolower($transaction['status']); ?>">
                                        <?php echo htmlspecialchars($transaction['status']); ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script src="assets/js/admin.js"></script>
</body>
</html>
