<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

$database = new Database();
$conn = $database->getConnection();

$success_message = '';
$error_message = '';

// Ensure upload directories exist
$uploadsDir = __DIR__ . '/uploads/site/';
if (!is_dir($uploadsDir)) { @mkdir($uploadsDir, 0755, true); }

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'update_settings') {
        $site_name = trim($_POST['site_name'] ?? '');
        $mobile_number = trim($_POST['mobile_number'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $logo_path_db = null;

        // Handle logo upload if provided
        if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
            $ext = strtolower(pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION));
            $allowed = ['png','jpg','jpeg','gif','webp','svg'];
            if (in_array($ext, $allowed)) {
                $fileName = 'logo_' . time() . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
                $destFs = $uploadsDir . $fileName;
                if (move_uploaded_file($_FILES['logo']['tmp_name'], $destFs)) {
                    $logo_path_db = 'uploads/site/' . $fileName;
                }
            }
        }

        try {
            if ($logo_path_db) {
                $stmt = $conn->prepare("UPDATE general_settings SET site_name=?, mobile_number=?, address=?, logo_path=?, updated_at=NOW() WHERE id=1");
                $stmt->execute([$site_name, $mobile_number, $address, $logo_path_db]);
            } else {
                $stmt = $conn->prepare("UPDATE general_settings SET site_name=?, mobile_number=?, address=?, updated_at=NOW() WHERE id=1");
                $stmt->execute([$site_name, $mobile_number, $address]);
            }
            $success_message = 'Settings updated successfully';
        } catch (Throwable $e) {
            $error_message = 'Failed to update settings: ' . $e->getMessage();
        }
    }

    if (isset($_POST['action']) && $_POST['action'] === 'add_hero') {
        if (isset($_FILES['hero_image']) && $_FILES['hero_image']['error'] === UPLOAD_ERR_OK) {
            $ext = strtolower(pathinfo($_FILES['hero_image']['name'], PATHINFO_EXTENSION));
            $allowed = ['png','jpg','jpeg','gif','webp'];
            if (in_array($ext, $allowed)) {
                $fileName = 'hero_' . time() . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
                $destFs = $uploadsDir . $fileName;
                if (move_uploaded_file($_FILES['hero_image']['tmp_name'], $destFs)) {
                    $title = trim($_POST['hero_title'] ?? '');
                    $stmt = $conn->prepare("INSERT INTO hero_images (image_path, title) VALUES (?, ?)");
                    $stmt->execute(['uploads/site/' . $fileName, $title]);
                    $success_message = 'Hero image added';
                } else {
                    $error_message = 'Failed to save hero image file';
                }
            } else {
                $error_message = 'Invalid hero image type';
            }
        } else {
            $error_message = 'No hero image uploaded';
        }
    }

    if (isset($_POST['action']) && $_POST['action'] === 'delete_hero') {
        $hero_id = (int)($_POST['hero_id'] ?? 0);
        if ($hero_id > 0) {
            $stmt = $conn->prepare("SELECT image_path FROM hero_images WHERE id = ?");
            $stmt->execute([$hero_id]);
            $row = $stmt->fetch();
            $stmt = $conn->prepare("DELETE FROM hero_images WHERE id = ?");
            $stmt->execute([$hero_id]);
            if ($row && !empty($row['image_path'])) {
                $fsPath = __DIR__ . '/' . $row['image_path'];
                if (is_file($fsPath)) { @unlink($fsPath); }
            }
            $success_message = 'Hero image deleted';
        }
    }
}

$settings = $conn->query("SELECT * FROM general_settings WHERE id=1")->fetch();
$heroes = $conn->query("SELECT * FROM hero_images ORDER BY created_at DESC")->fetchAll();

$database->closeConnection();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>General Settings</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .settings-wrap { max-width: 900px; margin: 24px auto; }
        .card { background:#fff; border-radius:12px; padding:20px; margin-bottom:18px; }
        .grid-2 { display:grid; grid-template-columns: 1fr 1fr; gap:16px; }
        .form-group label { display:block; margin-bottom:6px; font-weight:600; }
        .form-group input, .form-group textarea { width:100%; padding:10px; border:1px solid #e5e7eb; border-radius:8px; }
        .thumbs { display:grid; grid-template-columns: repeat(auto-fill, minmax(180px,1fr)); gap:14px; }
        .thumb { border:1px solid #e5e7eb; border-radius:10px; overflow:hidden; background:#fafafa; }
        .thumb img { width:100%; height:120px; object-fit:cover; display:block; }
        .thumb .meta { padding:10px; display:flex; justify-content:space-between; align-items:center; }
    </style>
</head>
<body>
    <div class="admin-container">
        <header class="admin-header">
            <div class="header-left">
                <h1 class="logo">TrustBeam Credit</h1>
                <button class="menu-toggle" id="menuToggle">
                    <i class="fas fa-bars"></i>
                </button>
            </div>
            <div class="header-right">
                <div class="admin-profile">
                    <span>Welcome, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
                    <div class="profile-dropdown">
                        <i class="fas fa-user"></i>
                        <div class="dropdown-content">
                            <a href="profile.php">Profile</a>
                            <a href="logout.php">Logout</a>
                        </div>
                    </div>
                </div>
            </div>
        </header>

        <aside class="admin-sidebar" id="sidebar">
            <div class="sidebar-header">
                <h3>Admin Manager</h3>
                <p>Admin</p>
            </div>
            <nav class="sidebar-nav">
                <ul>
                    <li><a href="index.php"><i class="fas fa-home"></i><span>Dashboard</span></a></li>
                    <li><a href="users.php"><i class="fas fa-users"></i><span>Manage Users</span></a></li>
                    <li class="active"><a href="settings.php"><i class="fas fa-cog"></i><span>General Settings</span></a></li>
                    <li><a href="email.php"><i class="fas fa-envelope"></i><span>Email Services</span></a></li>
                </ul>
            </nav>
        </aside>

        <main class="admin-main">
            <div class="page-header">
                <h2>General Settings</h2>
            </div>

            <div class="settings-wrap">
                <?php if ($success_message): ?>
                    <div class="alert alert-success"><i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success_message); ?></div>
                <?php endif; ?>
                <?php if ($error_message): ?>
                    <div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_message); ?></div>
                <?php endif; ?>

                <div class="card">
                    <h3>Site Information</h3>
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="update_settings">
                        <div class="grid-2">
                            <div class="form-group">
                                <label for="site_name">Site Name</label>
                                <input type="text" id="site_name" name="site_name" value="<?php echo htmlspecialchars($settings['site_name'] ?? ''); ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="mobile_number">Mobile Number</label>
                                <input type="text" id="mobile_number" name="mobile_number" value="<?php echo htmlspecialchars($settings['mobile_number'] ?? ''); ?>">
                            </div>
                        </div>
                        <div class="form-group">
                            <label for="address">Address</label>
                            <textarea id="address" name="address" rows="3"><?php echo htmlspecialchars($settings['address'] ?? ''); ?></textarea>
                        </div>
                        <div class="form-group">
                            <label for="logo">Logo (optional)</label>
                            <input type="file" id="logo" name="logo" accept="image/*">
                            <?php if (!empty($settings['logo_path'])): ?>
                                <div style="margin-top:8px;"><img src="<?php echo htmlspecialchars($settings['logo_path']); ?>" alt="Logo" style="height:48px;"></div>
                            <?php endif; ?>
                        </div>
                        <div class="form-actions">
                            <button type="submit" class="btn-primary"><i class="fas fa-save"></i> Save</button>
                        </div>
                    </form>
                </div>

                <div class="card">
                    <h3>Hero Images</h3>
                    <form method="POST" enctype="multipart/form-data" style="margin-bottom:16px;">
                        <input type="hidden" name="action" value="add_hero">
                        <div class="grid-2">
                            <div class="form-group">
                                <label for="hero_image">Upload Image</label>
                                <input type="file" id="hero_image" name="hero_image" accept="image/*" required>
                            </div>
                            <div class="form-group">
                                <label for="hero_title">Title (optional)</label>
                                <input type="text" id="hero_title" name="hero_title" placeholder="e.g., Secure Digital Banking">
                            </div>
                        </div>
                        <div class="form-actions">
                            <button type="submit" class="btn-primary"><i class="fas fa-upload"></i> Add Image</button>
                        </div>
                    </form>

                    <div class="thumbs">
                        <?php foreach ($heroes as $hero): ?>
                            <div class="thumb">
                                <img src="<?php echo htmlspecialchars($hero['image_path']); ?>" alt="Hero">
                                <div class="meta">
                                    <div><?php echo htmlspecialchars($hero['title'] ?? ''); ?></div>
                                    <form method="POST" onsubmit="return confirm('Delete this image?');">
                                        <input type="hidden" name="action" value="delete_hero">
                                        <input type="hidden" name="hero_id" value="<?php echo (int)$hero['id']; ?>">
                                        <button type="submit" class="btn-danger"><i class="fas fa-trash"></i></button>
                                    </form>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        <?php if (empty($heroes)): ?>
                            <div>No hero images yet.</div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>
    <script src="assets/js/admin.js"></script>
</body>
</html>


