// Cards Page JavaScript
// This handles all virtual cards functionality

document.addEventListener('DOMContentLoaded', function() {
    console.log('Cards page loaded - initializing...');
    initCardsPage();
});

function initCardsPage() {
    console.log('Initializing cards page...');
    
    // Initialize modal functionality
    initModalHandlers();
    
    // Initialize card interactions
    initCardInteractions();
    
    console.log('Cards page initialization complete');
}

function initModalHandlers() {
    // Close modal when clicking outside
    window.onclick = function(event) {
        const modal = document.getElementById('cardDetailsModal');
        if (event.target === modal) {
            modal.style.display = 'none';
        }
    }

    // Close modal when clicking close button
    const closeBtn = document.querySelector('.close');
    if (closeBtn) {
        closeBtn.onclick = function() {
            document.getElementById('cardDetailsModal').style.display = 'none';
        }
    }
}

function initCardInteractions() {
    // Add hover effects to cards
    const cardItems = document.querySelectorAll('.card-item');
    cardItems.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            this.style.boxShadow = '0 10px 25px rgba(0, 0, 0, 0.15)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
            this.style.boxShadow = '0 5px 15px rgba(0, 0, 0, 0.1)';
        });
    });
}

function showCardDetails(cardId) {
    console.log('Showing card details for card ID:', cardId);
    
    // Show modal
    const modal = document.getElementById('cardDetailsModal');
    modal.style.display = 'block';
    
    // Show loading state
    const content = document.getElementById('cardDetailsContent');
    content.innerHTML = '<div class="loading">Loading card details...</div>';
    
    // In a real application, you would fetch card details via AJAX
    // For now, we'll show a placeholder
    setTimeout(() => {
        content.innerHTML = `
            <div class="card-details-view">
                <div class="detail-section">
                    <h4>Card Information</h4>
                    <div class="detail-item">
                        <span class="detail-label">Card ID:</span>
                        <span class="detail-value">${cardId}</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Status:</span>
                        <span class="detail-value">Active</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Type:</span>
                        <span class="detail-value">Virtual Debit</span>
                    </div>
                </div>
                
                <div class="detail-section">
                    <h4>Security Information</h4>
                    <div class="detail-item">
                        <span class="detail-label">CVV:</span>
                        <span class="detail-value">***</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">PIN:</span>
                        <span class="detail-value">****</span>
                    </div>
                </div>
                
                <div class="detail-section">
                    <h4>Usage Limits</h4>
                    <div class="detail-item">
                        <span class="detail-label">Daily Limit:</span>
                        <span class="detail-value">$1,000.00</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Monthly Limit:</span>
                        <span class="detail-value">$5,000.00</span>
                    </div>
                </div>
                
                <div class="detail-section">
                    <h4>Transaction History</h4>
                    <div class="detail-item">
                        <span class="detail-label">Last Used:</span>
                        <span class="detail-value">Not used yet</span>
                    </div>
                    <div class="detail-item">
                        <span class="detail-label">Total Transactions:</span>
                        <span class="detail-value">0</span>
                    </div>
                </div>
            </div>
        `;
    }, 500);
}

// Utility functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'USD'
    }).format(amount);
}

function formatDate(date) {
    return new Date(date).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

// Show success message
function showSuccessMessage(message) {
    const alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-success';
    alertDiv.innerHTML = `
        <i class="fas fa-check-circle"></i>
        ${message}
    `;
    
    const pageHeader = document.querySelector('.page-header');
    if (pageHeader) {
        pageHeader.parentNode.insertBefore(alertDiv, pageHeader.nextSibling);
        
        // Remove after 5 seconds
        setTimeout(() => {
            alertDiv.remove();
        }, 5000);
    }
}

// Show error message
function showErrorMessage(message) {
    const alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-error';
    alertDiv.innerHTML = `
        <i class="fas fa-exclamation-circle"></i>
        ${message}
    `;
    
    const pageHeader = document.querySelector('.page-header');
    if (pageHeader) {
        pageHeader.parentNode.insertBefore(alertDiv, pageHeader.nextSibling);
        
        // Remove after 5 seconds
        setTimeout(() => {
            alertDiv.remove();
        }, 5000);
    }
}

// Confirm action
function confirmAction(message) {
    return confirm(message);
}

// Debug logging
console.log('Cards.js loaded successfully');
console.log('Available functions:', Object.keys(window).filter(key => typeof window[key] === 'function' && key.startsWith('show')));
