// Logout Confirmation JavaScript
// This script provides logout confirmation functionality across all pages

document.addEventListener('DOMContentLoaded', function() {
    initLogoutConfirmation();
});

// Initialize logout confirmation
function initLogoutConfirmation() {
    const logoutLinks = document.querySelectorAll('a[href*="logout.php"]');
    
    logoutLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            showLogoutConfirmation(link.href);
        });
    });
    
    console.log('Logout confirmation initialized');
}

// Show logout confirmation dialog
function showLogoutConfirmation(logoutUrl) {
    const modal = document.createElement('div');
    modal.className = 'logout-modal-overlay';
    modal.innerHTML = `
        <div class="logout-modal-content">
            <div class="logout-modal-header">
                <h3><i class="fas fa-sign-out-alt"></i> Confirm Logout</h3>
            </div>
            <div class="logout-modal-body">
                <div class="logout-message">
                    <div class="logout-icon">
                        <i class="fas fa-question-circle"></i>
                    </div>
                    <div class="logout-text">
                        <p><strong>Are you sure you want to logout?</strong></p>
                        <p>You will need to sign in again to access your account.</p>
                    </div>
                </div>
            </div>
            <div class="logout-modal-footer">
                <button class="logout-btn logout-btn-cancel" onclick="closeLogoutModal()">
                    <i class="fas fa-times"></i> No, Cancel
                </button>
                <button class="logout-btn logout-btn-confirm" onclick="confirmLogout('${logoutUrl}')">
                    <i class="fas fa-sign-out-alt"></i> Yes, Logout
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Close on outside click
    modal.addEventListener('click', function(event) {
        if (event.target === modal) {
            closeLogoutModal();
        }
    });
    
    // Close on escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            closeLogoutModal();
        }
    });
}

// Close logout modal
function closeLogoutModal() {
    const modal = document.querySelector('.logout-modal-overlay');
    if (modal) {
        modal.remove();
    }
}

// Confirm logout and redirect
function confirmLogout(logoutUrl) {
    closeLogoutModal();
    
    // Show loading state
    const loadingDiv = document.createElement('div');
    loadingDiv.className = 'logout-loading-overlay';
    loadingDiv.innerHTML = `
        <div class="logout-loading-content">
            <i class="fas fa-spinner fa-spin"></i>
            <p>Logging out...</p>
        </div>
    `;
    document.body.appendChild(loadingDiv);
    
    // Redirect after short delay
    setTimeout(() => {
        window.location.href = logoutUrl;
    }, 1000);
}

// Add CSS styles
const logoutStyles = document.createElement('style');
logoutStyles.textContent = `
    /* Logout Confirmation Modal Styles */
    .logout-modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.6);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10001;
        animation: logoutFadeIn 0.3s ease;
    }
    
    .logout-modal-content {
        background: white;
        border-radius: 12px;
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.3);
        max-width: 450px;
        width: 90%;
        max-height: 90%;
        overflow: hidden;
        animation: logoutSlideIn 0.3s ease;
    }
    
    .logout-modal-header {
        padding: 25px 30px 20px;
        border-bottom: 1px solid #e9ecef;
        background: linear-gradient(135deg, #fff5f5 0%, #ffe6e6 100%);
    }
    
    .logout-modal-header h3 {
        color: #dc3545;
        margin: 0;
        display: flex;
        align-items: center;
        gap: 12px;
        font-size: 1.3rem;
        font-weight: 600;
    }
    
    .logout-modal-body {
        padding: 30px;
    }
    
    .logout-message {
        display: flex;
        align-items: center;
        gap: 20px;
    }
    
    .logout-icon {
        width: 70px;
        height: 70px;
        border-radius: 50%;
        background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        color: #856404;
        font-size: 1.8rem;
        flex-shrink: 0;
        box-shadow: 0 4px 15px rgba(133, 100, 4, 0.2);
    }
    
    .logout-text {
        flex: 1;
    }
    
    .logout-text p {
        margin: 0 0 12px 0;
        line-height: 1.6;
        font-size: 1rem;
    }
    
    .logout-text p:first-child {
        font-size: 1.1rem;
        color: #2c3e50;
    }
    
    .logout-text p:last-child {
        margin-bottom: 0;
        color: #6c757d;
        font-size: 0.95rem;
    }
    
    .logout-modal-footer {
        display: flex;
        gap: 15px;
        justify-content: flex-end;
        padding: 20px 30px 25px;
        border-top: 1px solid #e9ecef;
        background: #f8f9fa;
    }
    
    .logout-btn {
        padding: 12px 24px;
        border: none;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 8px;
        font-size: 0.95rem;
        min-width: 120px;
        justify-content: center;
    }
    
    .logout-btn-cancel {
        background: #6c757d;
        color: white;
    }
    
    .logout-btn-cancel:hover {
        background: #5a6268;
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(108, 117, 125, 0.3);
    }
    
    .logout-btn-confirm {
        background: #dc3545;
        color: white;
    }
    
    .logout-btn-confirm:hover {
        background: #c82333;
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
    }
    
    /* Logout Loading Styles */
    .logout-loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10002;
        animation: logoutFadeIn 0.3s ease;
    }
    
    .logout-loading-content {
        background: white;
        padding: 50px;
        border-radius: 15px;
        text-align: center;
        box-shadow: 0 15px 35px rgba(0, 0, 0, 0.3);
        animation: logoutSlideIn 0.3s ease;
    }
    
    .logout-loading-content i {
        font-size: 2.5rem;
        color: #667eea;
        margin-bottom: 20px;
    }
    
    .logout-loading-content p {
        margin: 0;
        font-size: 1.2rem;
        color: #2c3e50;
        font-weight: 500;
    }
    
    /* Animations */
    @keyframes logoutFadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
    }
    
    @keyframes logoutSlideIn {
        from { 
            transform: translateY(-30px) scale(0.95);
            opacity: 0;
        }
        to { 
            transform: translateY(0) scale(1);
            opacity: 1;
        }
    }
    
    /* Mobile Responsive */
    @media (max-width: 768px) {
        .logout-modal-content {
            width: 95%;
            margin: 20px;
        }
        
        .logout-modal-header,
        .logout-modal-body,
        .logout-modal-footer {
            padding: 20px;
        }
        
        .logout-message {
            flex-direction: column;
            text-align: center;
            gap: 15px;
        }
        
        .logout-icon {
            width: 60px;
            height: 60px;
            font-size: 1.5rem;
        }
        
        .logout-modal-footer {
            flex-direction: column;
            gap: 12px;
        }
        
        .logout-btn {
            width: 100%;
            min-width: auto;
        }
    }
`;

document.head.appendChild(logoutStyles);
