// TrustBeam Credit Customer Website JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize all components
    initDropdowns();
    initMobileMenu();
    initDashboardMobileMenu();
    initScrollEffects();
    initFloatingButtons();
});

// Dropdown functionality
function initDropdowns() {
    const dropdowns = document.querySelectorAll('.dropdown');
    
    // Only proceed if dropdowns exist
    if (dropdowns.length === 0) {
        return;
    }
    
    dropdowns.forEach(dropdown => {
        const dropdownContent = dropdown.querySelector('.dropdown-content');
        
        // Only proceed if dropdown content exists
        if (!dropdownContent) {
            return;
        }
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function(event) {
            if (!dropdown.contains(event.target)) {
                dropdownContent.style.display = 'none';
            }
        });
        
        // Toggle dropdown on click for mobile
        dropdown.addEventListener('click', function(event) {
            if (window.innerWidth <= 768) {
                event.preventDefault();
                const isVisible = dropdownContent.style.display === 'block';
                dropdownContent.style.display = isVisible ? 'none' : 'block';
            }
        });
    });
}

// Mobile menu functionality
function initMobileMenu() {
    const navCenter = document.querySelector('.nav-center');
    const navRight = document.querySelector('.nav-right');
    const navContainer = document.querySelector('.nav-container');
    
    // Only proceed if the navigation elements exist (homepage)
    if (!navCenter || !navRight || !navContainer) {
        return;
    }
    
    if (window.innerWidth <= 768) {
        // Create mobile menu button
        const mobileMenuBtn = document.createElement('button');
        mobileMenuBtn.className = 'mobile-menu-btn';
        mobileMenuBtn.innerHTML = '<i class="fas fa-bars"></i>';
        mobileMenuBtn.style.display = 'block';
        
        // Insert mobile menu button
        navContainer.insertBefore(mobileMenuBtn, navCenter);
        
        // Mobile menu toggle
        mobileMenuBtn.addEventListener('click', function() {
            navCenter.classList.toggle('mobile-open');
            navRight.classList.toggle('mobile-open');
        });
        
        // Close mobile menu when clicking outside
        document.addEventListener('click', function(event) {
            if (!navContainer.contains(event.target)) {
                navCenter.classList.remove('mobile-open');
                navRight.classList.remove('mobile-open');
            }
        });
    }
}

// Dashboard mobile menu functionality
function initDashboardMobileMenu() {
    const mobileMenuToggle = document.getElementById('mobileMenuToggle');
    const dashboardSidebar = document.querySelector('.dashboard-sidebar');
    const sidebarOverlay = document.getElementById('sidebarOverlay');
    
    if (!mobileMenuToggle || !dashboardSidebar) return;
    
    // Toggle sidebar
    mobileMenuToggle.addEventListener('click', function() {
        dashboardSidebar.classList.toggle('active');
        if (sidebarOverlay) {
            sidebarOverlay.classList.toggle('active');
        }
        document.body.classList.toggle('sidebar-open');
    });
    
    // Close sidebar when clicking overlay
    if (sidebarOverlay) {
        sidebarOverlay.addEventListener('click', function() {
            dashboardSidebar.classList.remove('active');
            sidebarOverlay.classList.remove('active');
            document.body.classList.remove('sidebar-open');
        });
    }
    
    // Close sidebar when clicking outside
    document.addEventListener('click', function(e) {
        if (!dashboardSidebar.contains(e.target) && 
            !mobileMenuToggle.contains(e.target) && 
            dashboardSidebar.classList.contains('active')) {
            dashboardSidebar.classList.remove('active');
            if (sidebarOverlay) {
                sidebarOverlay.classList.remove('active');
            }
            document.body.classList.remove('sidebar-open');
        }
    });
    
    // Close sidebar on window resize if screen becomes large
    window.addEventListener('resize', function() {
        if (window.innerWidth > 768) {
            dashboardSidebar.classList.remove('active');
            if (sidebarOverlay) {
                sidebarOverlay.classList.remove('active');
            }
            document.body.classList.remove('sidebar-open');
        }
    });
}

// Scroll effects
function initScrollEffects() {
    const nav = document.querySelector('.main-nav');
    
    // Only proceed if the main navigation exists (homepage)
    if (!nav) {
        return;
    }
    
    let lastScrollTop = 0;
    
    window.addEventListener('scroll', function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        // Add shadow on scroll
        if (scrollTop > 50) {
            nav.style.boxShadow = '0 4px 20px rgba(0, 0, 0, 0.15)';
        } else {
            nav.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.1)';
        }
        
        // Hide/show navbar on scroll (mobile only)
        if (window.innerWidth <= 768) {
            if (scrollTop > lastScrollTop && scrollTop > 100) {
                nav.style.transform = 'translateY(-100%)';
            } else {
                nav.style.transform = 'translateY(0)';
            }
        }
        
        lastScrollTop = scrollTop;
    });
}

// Floating buttons functionality
function initFloatingButtons() {
    const floatingBtns = document.querySelectorAll('.floating-btn');
    
    // Only proceed if floating buttons exist
    if (floatingBtns.length === 0) {
        return;
    }
    
    floatingBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const action = btn.querySelector('span')?.textContent || 'Action';
            
            // Add click animation
            btn.style.transform = 'scale(0.95)';
            setTimeout(() => {
                btn.style.transform = 'scale(1)';
            }, 150);
            
            // Handle different actions
            switch(action) {
                case 'Make Payment':
                    showPaymentModal();
                    break;
                case 'Make an Enquiry':
                    showEnquiryModal();
                    break;
                default:
                    console.log('Action:', action);
            }
        });
    });
}

// Payment modal
function showPaymentModal() {
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-content">
            <div class="modal-header">
                <h3>Make Payment</h3>
                <button class="modal-close">&times;</button>
            </div>
            <div class="modal-body">
                <p>Payment functionality will be implemented here.</p>
                <p>This will connect to the banking backend for secure transactions.</p>
            </div>
            <div class="modal-footer">
                <button class="btn btn-secondary" onclick="closeModal()">Close</button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Close modal functionality
    const closeBtn = modal.querySelector('.modal-close');
    closeBtn.addEventListener('click', closeModal);
    
    // Close on outside click
    modal.addEventListener('click', function(event) {
        if (event.target === modal) {
            closeModal();
        }
    });
}

// Enquiry modal
function showEnquiryModal() {
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-content">
            <div class="modal-header">
                <h3>Make an Enquiry</h3>
                <button class="modal-close">&times;</button>
            </div>
            <div class="modal-body">
                <p>Enquiry functionality will be implemented here.</p>
                <p>Customers can submit questions and get support from the banking team.</p>
            </div>
            <div class="modal-footer">
                <button class="btn btn-secondary" onclick="closeModal()">Close</button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Close modal functionality
    const closeBtn = modal.querySelector('.modal-close');
    closeBtn.addEventListener('click', closeModal);
    
    // Close on outside click
    modal.addEventListener('click', function(event) {
        if (event.target === modal) {
            closeModal();
        }
    });
}

// Close modal function
function closeModal() {
    const modal = document.querySelector('.modal-overlay');
    if (modal) {
        modal.remove();
    }
}

// Smooth scrolling for anchor links
document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }
    });
});

// Form validation
function validateForm(form) {
    const inputs = form.querySelectorAll('input[required], textarea[required]');
    let isValid = true;
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            input.classList.add('error');
            isValid = false;
        } else {
            input.classList.remove('error');
        }
    });
    
    return isValid;
}

// Add CSS for modals
const modalStyles = document.createElement('style');
modalStyles.textContent = `
    .modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
    }
    
    .modal-content {
        background: white;
        border-radius: 12px;
        max-width: 500px;
        width: 90%;
        max-height: 80vh;
        overflow-y: auto;
        box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
    }
    
    .modal-header {
        padding: 20px 25px;
        border-bottom: 1px solid #e9ecef;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .modal-header h3 {
        margin: 0;
        color: #333;
    }
    
    .modal-close {
        background: none;
        border: none;
        font-size: 1.5rem;
        color: #6c757d;
        cursor: pointer;
        padding: 0;
        width: 30px;
        height: 30px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        transition: background-color 0.3s ease;
    }
    
    .modal-close:hover {
        background-color: #f8f9fa;
    }
    
    .modal-body {
        padding: 25px;
        color: #6c757d;
        line-height: 1.6;
    }
    
    .modal-footer {
        padding: 20px 25px;
        border-top: 1px solid #e9ecef;
        display: flex;
        justify-content: flex-end;
        gap: 10px;
    }
    
    .btn-secondary {
        background: #6c757d;
        color: white;
        border: none;
        padding: 10px 20px;
        border-radius: 6px;
        cursor: pointer;
        font-weight: 500;
        transition: background-color 0.3s ease;
    }
    
    .btn-secondary:hover {
        background: #5a6268;
    }
    
    .error {
        border-color: #dc3545 !important;
    }
    
    @media (max-width: 768px) {
        .mobile-menu-btn {
            display: block !important;
            background: none;
            border: none;
            font-size: 1.2rem;
            color: #333;
            cursor: pointer;
            padding: 10px;
        }
        
        .nav-center,
        .nav-right {
            display: none;
            width: 100%;
            flex-direction: column;
            gap: 10px;
            padding: 20px 0;
        }
        
        .nav-center.mobile-open,
        .nav-right.mobile-open {
            display: flex;
        }
        
        .nav-container {
            flex-wrap: wrap;
        }
    }
`;

document.head.appendChild(modalStyles);
