<?php
session_start();

// Check if user is already logged in
if (isset($_SESSION['user_id'])) {
    header('Location: dashboard.php');
    exit();
}

$error_message = '';
$success_message = '';

// Get general settings for logo and site name
$siteName = 'TrustBNK';
$logoPath = '';

try {
    require_once 'admin/config/database.php';
    $database = new Database();
    $db = $database->getConnection();
    
    $stmt = $db->prepare("SELECT site_name, logo_path FROM general_settings WHERE id = 1");
    $stmt->execute();
    $settings = $stmt->fetch();
    
    if ($settings) {
        $siteName = $settings['site_name'] ?: 'TrustBNK';
        $logoPath = $settings['logo_path'] ?: '';
        
        // Check if logo file exists
        if ($logoPath && !file_exists($logoPath)) {
            // Try alternative path
            $altPath = 'admin/' . $logoPath;
            if (file_exists($altPath)) {
                $logoPath = $altPath;
            } else {
                $logoPath = '';
            }
        }
    }
    
    $database->closeConnection();
} catch (Exception $e) {
    // Use default values if database connection fails
    $siteName = 'TrustBNK';
    $logoPath = '';
}

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    require_once 'admin/config/database.php';
    
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    
    if (empty($username) || empty($password)) {
        $error_message = "Please enter both username and password.";
    } else {
        try {
            $database = new Database();
            $conn = $database->getConnection();
            
            // Check if user exists and is active
            $stmt = $conn->prepare("SELECT id, username, firstname, lastname, email, password_hash, status, account_number, balance, currency FROM users WHERE username = ? AND status = 'Active'");
            $stmt->execute([$username]);
            $user = $stmt->fetch();
            
            if ($user && password_verify($password, $user['password_hash'])) {
                // Login successful
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['firstname'] = $user['firstname'];
                $_SESSION['lastname'] = $user['lastname'];
                $_SESSION['email'] = $user['email'];
                $_SESSION['account_number'] = $user['account_number'];
                $_SESSION['balance'] = $user['balance'];
                $_SESSION['currency'] = $user['currency'];
                $_SESSION['user_type'] = 'customer';
                
                // Update last login
                $stmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                $stmt->execute([$user['id']]);
                
                $database->closeConnection();
                
                // Redirect to dashboard
                header('Location: dashboard.php');
                exit();
            } else {
                $error_message = "Invalid username or password, or account is not active.";
            }
            
            $database->closeConnection();
        } catch (Exception $e) {
            $error_message = "Login error. Please try again.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - <?php echo htmlspecialchars($siteName); ?> Credit</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    
    <!-- Google Translator -->
    <script type="text/javascript">
        function googleTranslateElementInit() {
            new google.translate.TranslateElement({
                pageLanguage: 'en',
                includedLanguages: 'en,es,fr,de,it,pt,ru,ja,ko,zh-CN,zh-TW,ar,hi',
                layout: google.translate.TranslateElement.InlineLayout.SIMPLE,
                autoDisplay: false
            }, 'google_translate_element');
        }
    </script>
    <script type="text/javascript" src="//translate.google.com/translate_a/element.js?cb=googleTranslateElementInit"></script>
    
    <style>
        /* Google Translator Styles - Fixed Position Bottom */
        .translator-container {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 50px;
            padding: 10px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
        }
        
        .translator-container:hover {
            background: rgba(255, 255, 255, 1);
            box-shadow: 0 12px 35px rgba(0, 0, 0, 0.2);
            transform: scale(1.05);
        }
        
        .goog-te-gadget {
            font-family: 'Inter', sans-serif !important;
        }
        
        .goog-te-gadget-simple {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
            border: none !important;
            border-radius: 25px !important;
            padding: 12px 20px !important;
            font-size: 0.9rem !important;
            color: white !important;
            cursor: pointer !important;
            transition: all 0.3s ease !important;
            font-weight: 600 !important;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3) !important;
        }
        
        .goog-te-gadget-simple:hover {
            background: linear-gradient(135deg, #764ba2 0%, #667eea 100%) !important;
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4) !important;
            transform: translateY(-2px) !important;
        }
        
        .goog-te-gadget-simple .goog-te-menu-value span:first-child {
            color: white !important;
        }
        
        .goog-te-gadget-simple .goog-te-menu-value span:first-child:after {
            content: " 🌐" !important;
        }
        
        .goog-te-banner-frame {
            display: none !important;
        }
        
        .goog-te-menu2 {
            background: white !important;
            border: 1px solid #ddd !important;
            border-radius: 12px !important;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2) !important;
            margin-top: 10px !important;
            overflow: hidden !important;
        }
        
        .goog-te-menu2-item {
            padding: 12px 20px !important;
            border-bottom: 1px solid #f0f0f0 !important;
            transition: all 0.3s ease !important;
        }
        
        .goog-te-menu2-item:hover {
            background: #f8f9fa !important;
            padding-left: 25px !important;
        }
        
        .goog-te-menu2-item:last-child {
            border-bottom: none !important;
        }
        
        @media (max-width: 768px) {
            .translator-container {
                bottom: 15px;
                right: 15px;
                padding: 8px;
            }
            
            .goog-te-gadget-simple {
                padding: 10px 16px !important;
                font-size: 0.8rem !important;
            }
        }
        
        @media (max-width: 480px) {
            .translator-container {
                bottom: 10px;
                right: 10px;
                padding: 6px;
            }
            
            .goog-te-gadget-simple {
                padding: 8px 12px !important;
                font-size: 0.75rem !important;
            }
        }
        
        /* Mobile Menu Styles */
        .mobile-menu-toggle {
            display: none;
            background: none;
            border: none;
            color: white;
            font-size: 1.5rem;
            cursor: pointer;
            padding: 10px;
            margin-left: 15px;
            transition: all 0.3s ease;
        }
        
        .mobile-menu-toggle:hover {
            color: #ffa502;
            transform: scale(1.1);
        }
        
        .mobile-nav {
            display: none;
            position: absolute;
            top: 100%;
            left: 0;
            width: 100%;
            background: rgba(102, 126, 234, 0.98);
            backdrop-filter: blur(10px);
            padding: 20px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
            z-index: 1000;
        }
        
        .mobile-nav.active {
            display: block;
            animation: slideDown 0.3s ease;
        }
        
        .mobile-nav a {
            display: block;
            color: white;
            text-decoration: none;
            padding: 12px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            transition: all 0.3s ease;
            font-weight: 500;
        }
        
        .mobile-nav a:hover {
            color: #ffa502;
            padding-left: 10px;
        }
        
        .mobile-nav a:last-child {
            border-bottom: none;
        }
        
        .mobile-nav .mobile-login-btn,
        .mobile-nav .mobile-register-btn {
            background: rgba(255, 255, 255, 0.1);
            border-radius: 6px;
            margin: 10px 0;
            padding: 12px 20px;
            text-align: center;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .mobile-nav .mobile-login-btn:hover,
        .mobile-nav .mobile-register-btn:hover {
            background: rgba(255, 255, 255, 0.2);
            border-color: rgba(255, 255, 255, 0.3);
            padding-left: 20px;
        }
        
        .mobile-nav .mobile-register-btn {
            background: linear-gradient(135deg, #ff4757, #ff6b7a);
            border-color: #ff4757;
        }
        
        .mobile-nav .mobile-register-btn:hover {
            background: linear-gradient(135deg, #ff6b7a, #ff4757);
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Responsive Header Styles */
        @media (max-width: 768px) {
            .mobile-menu-toggle {
                display: block;
            }
            
            .nav-center {
                display: none;
            }
            
            .nav-right .btn {
                display: none;
            }
            
            .nav-right .mobile-menu-toggle {
                display: block;
            }
            
            .top-header {
                flex-direction: column;
                padding: 10px 20px;
            }
            
            .top-header-left,
            .top-header-right {
                width: 100%;
                justify-content: center;
                margin: 5px 0;
            }
            
            .top-header-right {
                flex-wrap: wrap;
                gap: 10px;
            }
            
            .main-nav {
                padding: 15px 20px;
            }
            
            .nav-container {
                flex-wrap: wrap;
            }
            
            .nav-left {
                flex: 1;
            }
            
            .nav-right {
                flex: 0 0 auto;
            }
        }
        
        @media (max-width: 480px) {
            .top-header {
                padding: 8px 15px;
            }
            
            .top-header-left a,
            .top-header-right a {
                font-size: 0.9rem;
                padding: 8px 12px;
            }
            
            .main-nav {
                padding: 12px 15px;
            }
            
            .logo-text .logo-main {
                font-size: 1.2rem;
            }
            
            .logo-text .logo-sub {
                font-size: 0.8rem;
            }
            
            
            .mobile-nav {
                padding: 15px;
            }
            
            .mobile-nav a {
                padding: 10px 0;
                font-size: 0.95rem;
            }
        }
    </style>
</head>
<body class="login-page">
    <!-- Top Header Bar -->
    <div class="top-header">
        <div class="top-header-left">
            <a href="index.php" class="top-link">
                <i class="fas fa-arrow-left"></i> Back to Home
            </a>
        </div>
        <div class="top-header-right">
            <a href="register.php" class="top-link">Register</a>
            <a href="#" class="top-link">
                <i class="fas fa-search"></i> Search
            </a>
        </div>
    </div>

    <!-- Main Navigation Bar -->
    <nav class="main-nav">
        <div class="nav-container">
            <div class="nav-left">
                <div class="logo">
                    <div class="logo-emblem">
                        <span class="logo-t">T</span>
                        <span class="logo-b">B</span>
                    </div>
                    <div class="logo-text">
                        <div class="logo-main"><?php echo htmlspecialchars($siteName); ?></div>
                        <div class="logo-sub">Credit</div>
                    </div>
                </div>
            </div>
            
            <div class="nav-center">
                <a href="index.php" class="nav-link">Home</a>
                <a href="#" class="nav-link">Services</a>
                <a href="#" class="nav-link">About</a>
                <a href="#" class="nav-link">Apply Now</a>
                <a href="#" class="nav-link">Get In Touch</a>
            </div>
            
            <div class="nav-right">
                <a href="login.php" class="btn btn-login active">
                    <i class="fas fa-user"></i> Login
                </a>
                <a href="register.php" class="btn btn-primary">
                    <i class="fas fa-credit-card"></i> Open an Account
                </a>
                
                <!-- Mobile Menu Toggle -->
                <button class="mobile-menu-toggle" id="mobileMenuToggle">
                    <i class="fas fa-bars"></i>
                </button>
            </div>
        </div>
        
        <!-- Mobile Navigation -->
        <div class="mobile-nav" id="mobileNav">
            <a href="index.php" class="active">Home</a>
            <a href="#">Services</a>
            <a href="#">About</a>
            <a href="#">Apply Now</a>
            <a href="#">Get In Touch</a>
            <a href="login.php" class="mobile-login-btn">Login</a>
            <a href="register.php" class="mobile-register-btn">Open an Account</a>
        </div>
    </nav>

    <!-- Login Section -->
    <section class="login-section">
        <div class="container">
            <div class="login-container">
                <div class="login-left">
                    <div class="login-content">
                        <h1>Welcome Back to TrustBeam Credit</h1>
                        <p>Access your account securely and manage your finances with ease.</p>
                        
                        <div class="login-features">
                            <div class="feature-item">
                                <i class="fas fa-shield-alt"></i>
                                <span>Secure Banking</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-mobile-alt"></i>
                                <span>Mobile Access</span>
                            </div>
                            <div class="feature-item">
                                <i class="fas fa-clock"></i>
                                <span>24/7 Banking</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="login-right">
                    <div class="login-form-container">
                        <div class="login-header">
                            <h2>Sign In to Your Account</h2>
                            <p>Enter your credentials to access your banking dashboard</p>
                        </div>
                        
                        <?php if ($error_message): ?>
                            <div class="alert alert-error">
                                <i class="fas fa-exclamation-circle"></i>
                                <?php echo htmlspecialchars($error_message); ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($success_message): ?>
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle"></i>
                                <?php echo htmlspecialchars($success_message); ?>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" class="login-form">
                            <div class="form-group">
                                <label for="username">
                                    <i class="fas fa-user"></i> Username
                                </label>
                                <input type="text" id="username" name="username" 
                                       value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>"
                                       placeholder="Enter your username" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="password">
                                    <i class="fas fa-lock"></i> Password
                                </label>
                                <div class="password-input">
                                    <input type="password" id="password" name="password" 
                                           placeholder="Enter your password" required>
                                    <button type="button" class="password-toggle" onclick="togglePassword()">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="form-options">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="remember" value="1">
                                    <span class="checkmark"></span>
                                    Remember me
                                </label>
                                <a href="forgot-password.php" class="forgot-link">Forgot Password?</a>
                            </div>
                            
                            <button type="submit" class="btn btn-login-submit">
                                <i class="fas fa-sign-in-alt"></i>
                                Sign In
                            </button>
                        </form>
                        
                        <div class="login-footer">
                            <p>Don't have an account? <a href="register.php">Register Now</a></p>
                            <p class="secure-note">
                                <i class="fas fa-lock"></i>
                                Your login is secured with bank-grade encryption
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <div class="footer-content">
                <div class="footer-section">
                    <h4>TrustBeam Credit</h4>
                    <p>Building a brighter financial future for our customers with innovative banking solutions.</p>
                </div>
                <div class="footer-section">
                    <h4>Quick Links</h4>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="#">About Us</a></li>
                        <li><a href="#">Services</a></li>
                        <li><a href="#">Contact</a></li>
                    </ul>
                </div>
                <div class="footer-section">
                    <h4>Contact Info</h4>
                    <p><i class="fas fa-phone"></i> +1 (555) 123-4567</p>
                    <p><i class="fas fa-envelope"></i> info@trustbeamcredit.com</p>
                    <p><i class="fas fa-map-marker-alt"></i> 123 Banking Street, Financial District</p>
                </div>
            </div>
            <div class="footer-bottom">
                <p>&copy; 2025 TrustBeam Credit. All rights reserved.</p>
                <div class="footer-links">
                    <a href="#">Privacy Policy</a>
                    <a href="#">Terms of Service</a>
                    <a href="#">Contact Support</a>
                </div>
            </div>
        </div>
    </footer>

    <script>
        function togglePassword() {
            const passwordInput = document.getElementById('password');
            const toggleBtn = document.querySelector('.password-toggle i');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                toggleBtn.className = 'fas fa-eye-slash';
            } else {
                passwordInput.type = 'password';
                toggleBtn.className = 'fas fa-eye';
            }
        }
        
        // Mobile menu toggle functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuToggle = document.getElementById('mobileMenuToggle');
            const mobileNav = document.getElementById('mobileNav');
            
            if (mobileMenuToggle && mobileNav) {
                mobileMenuToggle.addEventListener('click', function() {
                    mobileNav.classList.toggle('active');
                    const icon = this.querySelector('i');
                    icon.classList.toggle('fa-bars');
                    icon.classList.toggle('fa-times');
                });
                
                // Close mobile menu when clicking outside
                document.addEventListener('click', function(event) {
                    if (!mobileMenuToggle.contains(event.target) && !mobileNav.contains(event.target)) {
                        mobileNav.classList.remove('active');
                        const icon = mobileMenuToggle.querySelector('i');
                        icon.classList.add('fa-bars');
                        icon.classList.remove('fa-times');
                    }
                });
                
                // Close mobile menu when window is resized to desktop
                window.addEventListener('resize', function() {
                    if (window.innerWidth > 768) {
                        mobileNav.classList.remove('active');
                        const icon = mobileMenuToggle.querySelector('i');
                        icon.classList.add('fa-bars');
                        icon.classList.remove('fa-times');
                    }
                });
            }
        });
    </script>
    
    <!-- Fixed Google Translator -->
    <div id="google_translate_element" class="translator-container"></div>
</body>
</html>
