<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'customer') {
    header('Location: login.php');
    exit();
}

// Check if transfer_id exists in session
if (!isset($_SESSION['transfer_id'])) {
    header('Location: transfer.php');
    exit();
}

require_once 'admin/config/database.php';

// Format currency display
function formatCurrency($amount, $currencySymbol) {
    return $currencySymbol . number_format($amount, 2);
}

$database = new Database();
$conn = $database->getConnection();

// Get user data
$user_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

// Get transfer data
$transfer_id = $_SESSION['transfer_id'];
$stmt = $conn->prepare("SELECT * FROM transfers WHERE id = ? AND user_id = ? AND transfer_status = 'IMF_Verified'");
$stmt->execute([$transfer_id, $user_id]);
$transfer = $stmt->fetch();

if (!$transfer) {
    unset($_SESSION['transfer_id']);
    header('Location: transfer.php');
    exit();
}

$verification_error = '';
$verification_success = false;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $swift_code = trim($_POST['swift_code'] ?? '');
    
    if (empty($swift_code)) {
        $verification_error = "SWIFT code is required";
    } elseif ($swift_code !== $user['swift_code']) {
        $verification_error = "Invalid SWIFT code. Please check and try again.";
    } else {
        // Update transfer status to SWIFT_Verified
        $stmt = $conn->prepare("UPDATE transfers SET transfer_status = 'SWIFT_Verified', swift_verified_at = NOW() WHERE id = ?");
        $stmt->execute([$transfer_id]);
        
        // Redirect to COT verification
        header('Location: transfer-verify-cot.php');
        exit();
    }
}

$database->closeConnection();

// Get current date and time
$current_date = date('l, F j, Y');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SWIFT Verification - TrustBeam Credit</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .verification-container {
            max-width: 600px;
            margin: 0 auto;
            padding: 20px;
        }

        .verification-card {
            background: white;
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border: 1px solid #e0e6ed;
            text-align: center;
        }

        .verification-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 30px;
            color: white;
            font-size: 2rem;
        }

        .verification-title {
            color: #2c3e50;
            margin: 0 0 15px 0;
            font-size: 1.8rem;
            font-weight: 600;
        }

        .verification-subtitle {
            color: #6c757d;
            margin: 0 0 30px 0;
            font-size: 1.1rem;
        }

        .transfer-details {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
            text-align: left;
        }

        .transfer-detail-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 10px;
        }

        .transfer-detail-row:last-child {
            margin-bottom: 0;
        }

        .transfer-detail-label {
            color: #6c757d;
            font-weight: 500;
        }

        .transfer-detail-value {
            color: #2c3e50;
            font-weight: 600;
        }

        .verification-form {
            margin-bottom: 30px;
        }

        .form-group {
            margin-bottom: 25px;
        }

        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: #2c3e50;
            font-size: 1rem;
        }

        .form-group input {
            width: 100%;
            padding: 15px 20px;
            border: 2px solid #e0e6ed;
            border-radius: 10px;
            font-size: 1.2rem;
            text-align: center;
            letter-spacing: 2px;
            font-family: 'Courier New', monospace;
            transition: border-color 0.3s ease;
        }

        .form-group input:focus {
            outline: none;
            border-color: #28a745;
        }

        .verification-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }

        .btn-verify {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border: none;
            padding: 15px 40px;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.3s ease;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .btn-verify:hover {
            transform: translateY(-2px);
        }

        .btn-cancel {
            background: #6c757d;
            color: white;
            border: none;
            padding: 15px 30px;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.3s ease;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .btn-cancel:hover {
            background: #5a6268;
        }


        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-weight: 500;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        /* Mobile responsive */
        @media (max-width: 768px) {
            .verification-container {
                padding: 10px;
            }
            
            .verification-card {
                padding: 30px 20px;
            }
            
            .verification-actions {
                flex-direction: column;
                align-items: stretch;
            }
        }
    </style>
</head>
<body class="dashboard-page">
    <!-- Top Header Bar -->
    <div class="dashboard-header">
        <div class="header-left">
            <div class="current-balance">
                <span class="balance-label">Current Balance</span>
                <span class="balance-amount"><?php echo formatCurrency($user['balance'], $_SESSION['currency'] ?? '$'); ?></span>
            </div>
        </div>
        <div class="header-center">
            <div class="current-date"><?php echo $current_date; ?></div>
        </div>
        <div class="header-right">
            <div class="header-actions">
                <button class="notification-btn">
                    <i class="fas fa-bell"></i>
                    <span class="notification-badge">3</span>
                </button>
                <div class="user-profile">
                    <div class="user-avatar">
                        <?php if ($user['profile_photo']): ?>
                            <img src="admin/<?php echo htmlspecialchars($user['profile_photo']); ?>" alt="Profile Photo">
                        <?php else: ?>
                            <span><?php echo strtoupper(substr($user['firstname'], 0, 1) . substr($user['lastname'], 0, 1)); ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="user-info">
                        <span class="user-name"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></span>
                        <span class="user-id"><?php echo htmlspecialchars($user['account_number']); ?></span>
                        <span class="kyc-badge">KYC Verified</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="dashboard-container">
        <!-- Mobile Menu Toggle -->
        <button class="mobile-menu-toggle" id="mobileMenuToggle">
            <i class="fas fa-bars"></i>
        </button>
        
        <!-- Sidebar Overlay -->
        <div class="sidebar-overlay" id="sidebarOverlay"></div>
        
        <!-- Sidebar -->
        <aside class="dashboard-sidebar">
            <div class="sidebar-header">
                <div class="logo">
                    <div class="logo-emblem">
                        <span class="logo-t">T</span>
                        <span class="logo-b">B</span>
                    </div>
                    <div class="logo-text">
                        <div class="logo-main">TrustBeam</div>
                        <div class="logo-sub">Credit</div>
                    </div>
                </div>
            </div>
            
            <div class="user-profile-sidebar">
                <div class="user-avatar-sidebar">
                    <?php if ($user['profile_photo']): ?>
                        <img src="admin/<?php echo htmlspecialchars($user['profile_photo']); ?>" alt="Profile Photo">
                    <?php else: ?>
                        <span><?php echo strtoupper(substr($user['firstname'], 0, 1) . substr($user['lastname'], 0, 1)); ?></span>
                    <?php endif; ?>
                </div>
                <div class="user-details">
                    <div class="user-name"><?php echo htmlspecialchars($user['firstname'] . ' ' . $user['lastname']); ?></div>
                    <div class="user-id"><?php echo htmlspecialchars($user['account_number']); ?></div>
                    <span class="kyc-badge">KYC Verified</span>
                </div>
                <div class="profile-actions">
                    <a href="profile.php" class="btn-profile">
                        <i class="fas fa-user"></i> Profile
                    </a>
                    <a href="logout.php" class="btn-logout">
                        <i class="fas fa-sign-out-alt"></i> Logout
                    </a>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-section">
                    <h3>Main Menu</h3>
                    <ul>
                        <li><a href="dashboard.php"><i class="fas fa-home"></i> Dashboard</a></li>
                        <li><a href="transactions.php"><i class="fas fa-exchange-alt"></i> Transactions</a></li>
                        <li><a href="transfer.php" class="active"><i class="fas fa-paper-plane"></i> Transfer Funds</a></li>
                        <li><a href="cards.php"><i class="fas fa-credit-card"></i> Cards</a></li>
                    </ul>
                </div>
                
                <div class="nav-section">
                    <h3>Services</h3>
                    <ul>
                        <li><a href="loan-request.php"><i class="fas fa-hand-holding-usd"></i> Loan Request</a></li>
                        <li><a href="loan-history.php"><i class="fas fa-history"></i> Loan History</a></li>
                        <li><a href="irs-refund.php"><i class="fas fa-file-invoice-dollar"></i> IRS Refund</a></li>
                        <li><a href="apply-card.php"><i class="fas fa-id-card"></i> Apply for Card</a></li>
                    </ul>
                </div>
            </nav>
            
            <div class="sidebar-footer">
                <div class="support-info">
                    <h4>Need Help?</h4>
                    <p>Contact our 24/7 support team</p>
                    <a href="mailto:support@trustbeamcredit.com" class="support-link">
                        <i class="fas fa-envelope"></i> support@trustbeamcredit.com
                    </a>
                </div>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="dashboard-main">
            <div class="verification-container">
                <div class="verification-card">
                    <div class="verification-icon">
                        <i class="fas fa-globe"></i>
                    </div>

                    <h1 class="verification-title">SWIFT Verification</h1>
                    <p class="verification-subtitle">Enter your SWIFT code to proceed with the transfer verification</p>

                    <!-- Transfer Details -->
                    <div class="transfer-details">
                        <div class="transfer-detail-row">
                            <span class="transfer-detail-label">Transfer Amount:</span>
                            <span class="transfer-detail-value"><?php echo formatCurrency($transfer['amount'], $_SESSION['currency'] ?? '$'); ?></span>
                        </div>
                        <div class="transfer-detail-row">
                            <span class="transfer-detail-label">To:</span>
                            <span class="transfer-detail-value"><?php echo htmlspecialchars($transfer['beneficiary_account_name']); ?></span>
                        </div>
                        <div class="transfer-detail-row">
                            <span class="transfer-detail-label">Reference:</span>
                            <span class="transfer-detail-value"><?php echo htmlspecialchars($transfer['reference_number']); ?></span>
                        </div>
                    </div>

                    <?php if ($verification_error): ?>
                        <div class="alert alert-error">
                            <i class="fas fa-exclamation-circle"></i>
                            <?php echo $verification_error; ?>
                        </div>
                    <?php endif; ?>

                    <!-- Verification Form -->
                    <form method="POST" class="verification-form">
                        <div class="form-group">
                            <label for="swift_code">Enter your SWIFT Code</label>
                            <input type="text" id="swift_code" name="swift_code" required maxlength="11" 
                                   placeholder="Enter SWIFT code" autocomplete="off" style="text-transform: uppercase;">
                        </div>

                        <div class="verification-actions">
                            <button type="submit" class="btn-verify">
                                <i class="fas fa-check"></i> Verify SWIFT Code
                            </button>
                            <a href="transfer.php" class="btn-cancel">
                                <i class="fas fa-times"></i> Cancel
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Mobile menu toggle
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuToggle = document.getElementById('mobileMenuToggle');
            const sidebar = document.querySelector('.dashboard-sidebar');
            const overlay = document.getElementById('sidebarOverlay');

            mobileMenuToggle.addEventListener('click', function() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
            });

            overlay.addEventListener('click', function() {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            });

            // Focus on SWIFT code input and auto-uppercase
            const swiftInput = document.getElementById('swift_code');
            swiftInput.focus();
            
            swiftInput.addEventListener('input', function() {
                this.value = this.value.toUpperCase();
            });
        });
    </script>
    <script src="assets/js/logout-confirmation.js"></script>
</body>
</html>
